/*
 *  Envlink File System, version 0.3, 01.02.1999
 *
 *  envlinkfs/init.c
 *
 *  Copyright (C) 1998-1999 by Stelian Pop <pop@cybercable.fr>
 *
 */

#include <linux/config.h>
#include <linux/module.h>
#include <linux/version.h>
#include <linux/fs.h>
#include <linux/mm.h>
#include <linux/errno.h>
#include <linux/string.h>
#include <linux/malloc.h>
#include <asm/segment.h>
#include <asm/uaccess.h>
#include <linux/locks.h>

#include "envlinkfs.h"

extern struct inode_operations envlinkfs_inode_operations;
extern struct inode_operations envlinkfs_dir_inode_operations;

/*
 * Fills the inode structure.
 */
static void 
envlinkfs_read_inode(struct inode *inode)
{
	unsigned long ino = inode->i_ino;

	PRINTK(KERN_DEBUG "envlinkfs: envlinkfs_read_inode\n");

	inode->i_op = NULL;
	inode->i_mode = 0;
	inode->i_uid = 0;
	inode->i_gid = 0;
	inode->i_nlink = 1;
	inode->i_size = 0;
	inode->i_mtime = inode->i_atime = inode->i_ctime = CURRENT_TIME;
	inode->i_blocks = 0;
	inode->i_blksize = 1024;

	switch (ino) {
		case ENVLINKFS_ROOT_INODE:
			inode->i_mode = S_IFDIR | S_IRUGO | S_IXUGO;
			inode->i_nlink = 2;
			inode->i_op = &envlinkfs_dir_inode_operations;
			break;
		default:
			inode->i_mode = S_IFLNK | S_IRUGO | S_IWUGO | S_IXUGO;
			inode->i_op = &envlinkfs_inode_operations;
	}
	return;
}

/*
 * Free the superblock.
 */
static void
envlinkfs_put_super(struct super_block *sb)
{
	PRINTK(KERN_DEBUG "envlinkfs: envlinkfs_put_super\n");

	MOD_DEC_USE_COUNT;
}

/*
 * File system statistics.
 */
static int 
envlinkfs_statfs(struct super_block *sb, struct statfs *buf, int bufsiz)
{
	struct statfs tmp;

	PRINTK(KERN_DEBUG "envlinkfs: envlinkfs_statfs\n");

	memset(&tmp, 0, sizeof(tmp));
	tmp.f_type = ENVLINKFS_SUPER_MAGIC;
	tmp.f_bsize = PAGE_SIZE/sizeof(long);
	tmp.f_blocks = 0;
	tmp.f_bfree = 0;
	tmp.f_bavail = 0;
	tmp.f_files = 0;
	tmp.f_ffree = 0;
	tmp.f_namelen = NAME_MAX;
	return copy_to_user(buf, &tmp, bufsiz) ? -EFAULT : 0;
}

/*
 * Superblock operations.
 */
static struct super_operations envlinkfs_sops = {
	envlinkfs_read_inode,	/* read inode */
	NULL,			/* write inode */
	NULL,			/* put inode */
	NULL,			/* delete inode */
	NULL,			/* notify change */
	envlinkfs_put_super,	/* put super */
	NULL,			/* write super */
	envlinkfs_statfs,	/* statfs */
	NULL			/* remount */
};

/*
 * Returns the inode referenced by its number.
 */
struct inode *
envlinkfs_get_inode(struct super_block *s, int ino)
{
	struct inode *inode;

	PRINTK(KERN_DEBUG "envlinkfs: envlinkfs_get_inode\n");

	inode = iget(s, ino);
	if (inode && inode->i_sb == s)
		envlinkfs_read_inode(inode);
	return inode;
}

/*
 * Initialise the superblock.
 */
struct super_block *
envlinkfs_read_super(struct super_block *s, void *data, int silent)
{
	struct inode *root_inode;

	PRINTK(KERN_DEBUG "envlinkfs: read_super\n");

	MOD_INC_USE_COUNT;
	lock_super(s);
	s->s_blocksize = 1024;
	s->s_blocksize_bits = 10;
	s->s_magic = ENVLINKFS_SUPER_MAGIC;
	s->s_op = &envlinkfs_sops;
	root_inode = envlinkfs_get_inode(s, ENVLINKFS_ROOT_INODE);
	if (!root_inode)
		goto out_no_root;
	s->s_root = d_alloc_root(root_inode, NULL);
	if (!s->s_root)
		goto out_no_root;	
	unlock_super(s);
	return s;

out_no_root:

	printk(KERN_ERR "envlinkfs: get root inode failed\n");

	iput(root_inode);
	s->s_dev = 0;
	unlock_super(s);
	MOD_DEC_USE_COUNT;
	return NULL;
}

static struct file_system_type envlinkfs_fs_type = {
	"envlinkfs",
	0,
	envlinkfs_read_super, 
	NULL
};

/*
 * Insmod-time options.
 */
char *options=NULL;
MODULE_PARM(options,"s");
MODULE_AUTHOR("Stelian Pop");

/*
 * Chained list of the directory entries.
 */
struct envlinkfs_dir_entry *de=NULL;

/*
 * Parses the options line and fills 'de'.
 */
int 
envlinkfs_read_options(void) {

	char *ptr1, *ptr2;
	struct envlinkfs_dir_entry *entry;
	unsigned int inode = ENVLINKFS_FIRST_INODE;

	if (!options)
		return -1;

	PRINTK(KERN_DEBUG "envlinkfs: envlinkfs_read_options: %s\n", options);

	ptr1 = strtok(options, " \t");
	if (!ptr1)
		return -1;
	ptr2 = strtok(NULL, " \t");
	if (!ptr2)
		return -1;

	while (1) {
		entry = (struct envlinkfs_dir_entry *)kmalloc(sizeof(struct envlinkfs_dir_entry), GFP_KERNEL);
		entry->ino = inode++;
		entry->name = kmalloc(strlen(ptr1)+1, GFP_KERNEL);
		if (*ptr1 == '$') {
			strcpy(entry->name, ptr1+1);
			entry->name_is_env = 1;
		}
		else {
			strcpy(entry->name, ptr1);
			entry->name_is_env = 0;
		}
		entry->dest = kmalloc(strlen(ptr2)+1, GFP_KERNEL);	
		if (*ptr2 == '$') {
			strcpy(entry->dest, ptr2+1);
			entry->dest_is_env = 1;
		}
		else {
			strcpy(entry->dest, ptr2);
			entry->dest_is_env = 0;
		}
		entry->next = de;
		de = entry;

		PRINTK(KERN_DEBUG "envlinkfs: added translation %s (%d) -> %s\n", ptr1, inode - 1, ptr2);

		ptr1 = strtok(NULL, " \t");
		if (!ptr1)
			return 0;
		ptr2 = strtok(NULL, " \t");
		if (!ptr2)
			return 0;
	}
	return 0;
}

/*
 * Free 'de'.
 */
void
envlinkfs_free(void)
{
	struct envlinkfs_dir_entry *entry;

	PRINTK(KERN_DEBUG "envlinkfs: envlinkfs_free\n");

	while (de) {

		entry = de->next;
		kfree(de->name);
		kfree(de->dest);
		kfree(de);
		de = entry;
	}
}

/*
 * Initialisation of the module.
 */
int
init_module(void)
{
	int err;

	PRINTK(KERN_DEBUG "envlinkfs: init_module\n");

	printk(KERN_INFO "envlinkfs: Envlinkfs 0.3, 01.02.1999, (C)1998-1999 Stelian Pop\n");

	err = envlinkfs_read_options();
	if (err) {
		printk(KERN_ERR "envlinkfs: incorrect options\n");
		return err;
	}

	err = register_filesystem(&envlinkfs_fs_type);
	if (err) {
		printk(KERN_ERR "envlinkfs: failed to register filesystem\n");
		return err;
	}

	printk(KERN_INFO "envlinkfs: successfully registered\n");
	return 0;
}

/*
 * Cleanup of the module.
 */
void 
cleanup_module(void)
{
	int err;

	PRINTK(KERN_DEBUG "envlinkfs: cleanup_module\n");

	envlinkfs_free();

	err = unregister_filesystem(&envlinkfs_fs_type);
	if (err) 
		printk(KERN_ERR "envlinkfs: failed to unregister filesystem\n");
	else
		printk(KERN_INFO "envlinkfs: successfully unregistered\n");
}
